<?php

namespace App\Http\Controllers;

use App\Http\Helpers\Common;
use App\Models\AppToken;
use App\Models\AppTransactionsInfo;
use App\Models\Currency;
use App\Models\MerchantApp;
use App\Models\MerchantPayment;
use App\Models\Preference;
use App\Models\Setting;
use App\Models\Transaction;
use App\Models\User;
use App\Models\Wallet;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;

class MerchantApiPayment extends Controller
{
    protected $helper;
    public function __construct()
    {
        $this->helper = new Common();
    }

    public function verifyClient(Request $request)
    {
        $app      = $this->verifyClientIdAndClientSecret($request->client_id, $request->client_secret);
        $response = $this->createAccessToken($app); //will expire in one hour
        return json_encode($response);
    }

    protected function verifyClientIdAndClientSecret($client_id, $client_secret)
    {
        $app = MerchantApp::where(['client_id' => $client_id, 'client_secret' => $client_secret])->first();
        if (!$app)
        {
            $res = [
                'status'  => 'error',
                'message' => 'Can not verify the client. Please check client Id and Client Secret',
                'data'    => [],
            ];
            return json_encode($res);
        }
        return $app;
    }

    protected function createAccessToken($app)
    {
        $token = $app->accessToken()->create(['token' => str_random(26), 'expires_in' => time() + 3600]);
        $res   = [
            'status'  => 'success',
            'message' => 'Client Verified',
            'data'    => [
                'access_token' => $token->token,
            ],
        ];
        return $res;
    }

    /**
     * [Generat URL]
     * @param  Request $request [email, password]
     * @return [view]           [redirect to merchant confirm page or redirect back]
     */
    public function generatedUrl(Request $request)
    {
        if (!auth()->check())
        {
            if ($_POST)
            {
                $credentials = $request->only('email', 'password');
                if (Auth::attempt($credentials))
                {
                    $this->setDefaultSessionValues();

                    $credentialsForConfirmPageLogin = [
                        'email'    => $request->email,
                        'password' => $request->password,
                    ];
                    Session::put('credentials', $credentialsForConfirmPageLogin);

                    //
                    $transInfo = AppTransactionsInfo::where(['grant_id' => $request->grant_id, 'token' => $request->token, 'status' => 'pending'])->where('expires_in', '>=', time())->first();
                    //Abort if logged in user is same as merchant
                    if ($transInfo->app->merchant->user->id == auth()->user()->id)
                    {
                        auth()->logout();
                        $this->helper->one_time_message('error', __('Merchant cannot make payment to himself!'));
                        return redirect()->back();
                    }
                    else
                    {
                        $data = $this->checkoutToPaymentConfirmPage($transInfo);
                        return view('merchantPayment.confirm', $data);
                    }
                    //
                }
                else
                {
                    $this->helper->one_time_message('error',  __('Unable to login with provided credentials!'));
                    return redirect()->back();
                }
            }
            else
            {
                $general         = Setting::where(['type' => 'general'])->get(['value', 'name'])->toArray();
                $data['setting'] = $setting = $this->helper->key_value('name', 'value', $general);
                return view('merchantPayment.login', $data);
            }
        }
        else
        {
            //
            $transInfo = AppTransactionsInfo::where(['grant_id' => $request->grant_id, 'token' => $request->token, 'status' => 'pending'])->where('expires_in', '>=', time())->first();
            //Abort if logged in user is same as merchant
            if ($transInfo->app->merchant->user->id == auth()->user()->id)
            {
                auth()->logout();
                $this->helper->one_time_message('error', __('Merchant cannot make payment to himself!'));
                return redirect()->back();
            }
            else
            {
                $data = $this->checkoutToPaymentConfirmPage($transInfo);
                // dd($data);
                return view('merchantPayment.confirm', $data);
            }
            //
        }
    }

    protected function checkoutToPaymentConfirmPage($transInfo)
    {
        //check expired or not
        if (!$transInfo)
        {
            abort(403, 'Url has been deleted or expired.');
        }

        //check if currency exists in wallets
        $availableCurrency = [];
        $wallets           = auth()->user()->wallets;
        foreach ($wallets as $wallet)
        {
            $availableCurrency[] = $wallet->currency->code;
        }
        if (!in_array($transInfo->currency, $availableCurrency))
        {
            $this->helper->one_time_message('error', "You don't have the payment wallet. Please create wallet for currency - {$transInfo->currency} !");
            return redirect()->to('payment/fail');
        }

        $data['currSymbol'] = $currSymbol = Currency::where('code', $transInfo->currency)->first(['symbol'])->symbol;
        $data['transInfo']  = $transInfo;

        //Put transaction informations to Session
        Session::put('transInfo', $transInfo);
        return $data;
    }

    public function storeTransactionInfo(Request $request)
    {
        // dd($request->successUrl);

        $paymentMethod = $request->payer;
        $amount        = $request->amount;
        $currency      = $request->currency;
        $successUrl    = $request->successUrl;
        $cancelUrl     = $request->cancelUrl;

        # check token missing
        $hasHeaderAuthorization = $request->hasHeader('Authorization');
        if (!$hasHeaderAuthorization)
        {
            $res = [
                'status'  => 'error',
                'message' => 'Access token is missing',
                'data'    => [],
            ];
            return json_encode($res);
        }

        # check token authorization
        $headerAuthorization = $request->header('Authorization');
        $token               = $this->checkTokenAuthorization($headerAuthorization);

        # Currency Validation
        $res = $this->currencyValidaation($token, $currency);
        if (!empty($res['status']))
        {
            return json_encode($res);
        }

        # Amount Validation
        $res = $this->amountValidaation($amount);
        if (!empty($res['status']))
        {
            return json_encode($res);
        }

        if (false)
        {
            $res = [
                'status'  => 'error',
                'message' => 'Validation error',
                'data'    => [],
            ];
            return json_encode($res);
        }

        # Update/Create AppTransactionsInfo and return response
        $res = $this->updateOrAppTransactionsInfoAndReturnResponse($token->app_id, $paymentMethod, $amount, $currency, $successUrl, $cancelUrl);
        return json_encode($res);
    }

    /**
     * [Set Necessary Values To Session]
     */
    protected function setDefaultSessionValues()
    {
        $preferences = Preference::where('field', '!=', 'dflt_lang')->get();
        if (!empty($preferences))
        {
            foreach ($preferences as $pref)
            {
                $pref_arr[$pref->field] = $pref->value;
            }
        }
        if (!empty($preferences))
        {
            Session::put($pref_arr);
        }

        // default_currency
        $default_currency = Setting::where('name', 'default_currency')->first();
        if (!empty($default_currency))
        {
            Session::put('default_currency', $default_currency->value);
        }

        //default_timezone
        $default_timezone = auth()->user()->user_detail->timezone;
        if (!$default_timezone)
        {
            Session::put('dflt_timezone_user', session('dflt_timezone'));
        }
        else
        {
            Session::put('dflt_timezone_user', $default_timezone);
        }

        // default_language
        $default_language = Setting::where('name', 'default_language')->first();
        if (!empty($default_language))
        {
            Session::put('default_language', $default_language->value);
        }

        // company_name
        $company_name = Setting::where('name', 'name')->first();
        if (!empty($company_name))
        {
            Session::put('name', $company_name->value);
        }

        // company_logo
        $company_logo = Setting::where(['name' => 'logo', 'type' => 'general'])->first();
        if (!empty($company_logo))
        {
            Session::put('company_logo', $company_logo->value);
        }
    }

    /**
     * [check Token Authorization]
     * @param  [request] $headerAuthorization [header authorization request]
     * @return [string]                      [token]
     */
    protected function checkTokenAuthorization($headerAuthorization)
    {
        $accessToken = $headerAuthorization;
        $tokenType   = '';
        $actualToken = '';
        if (preg_match('/\bBearer\b/', $accessToken))
        {
            $tokenType   = 'bearer';
            $t           = explode(' ', $accessToken);
            $key         = array_keys($t);
            $last        = end($key);
            $actualToken = $t[$last];
        }
        $token = AppToken::where('token', $actualToken)->where('expires_in', '>=', time())->first();
        if (!$token)
        {
            $res = [
                'status'  => 'error',
                'message' => 'Unauthorized token or token has been expired',
                'data'    => [],
            ];
            return json_encode($res);
        }
        return $token;
    }

    protected function currencyValidaation($token, $currency)
    {
        $acceptedCurrency = [];
        $wallets          = $token->app->merchant->user->wallets;
        foreach ($wallets as $wallet)
        {
            $acceptedCurrency[] = $wallet->currency->code;
        }
        //TODO:: Accepted currency will come from database or from merchant currency

        $res = ['status' => ''];
        if (!in_array($currency, $acceptedCurrency))
        {
            $res = [
                'status'  => 'error',
                'message' => 'Currency ' . $currency . ' is not supported by this merchant!',
                'data'    => [],
            ];
        }
        return $res;
    }

    protected function amountValidaation($amount)
    {
        $res = ['status' => ''];
        if ($amount <= 0)
        {
            $res = [
                'status'  => 'error',
                'message' => 'Amount cannot be 0 or less than 0.',
                'data'    => [],
            ];
        }
        return $res;
    }

    protected function updateOrAppTransactionsInfoAndReturnResponse($tokenAppId, $paymentMethod, $amount, $currency, $successUrl, $cancelUrl)
    {
        try
        {
            $grandId  = random_int(10000000, 99999999);
            $urlToken = str_random(20);

            AppTransactionsInfo::updateOrCreate([
                'app_id'         => $tokenAppId,
                'payment_method' => $paymentMethod,
                'amount'         => $amount,
                'currency'       => $currency,
                'success_url'    => $successUrl,
                'cancel_url'     => $cancelUrl,
                'grant_id'       => $grandId,
                'token'          => $urlToken,
                'expires_in'     => time() + (60 * 60 * 5), // url will expire in 5 hours after generation
            ]);

            $url = url("merchant/payment?grant_id=$grandId&token=$urlToken");
            $res = [
                'status'  => 'success',
                'message' => '',
                'data'    => [
                    'approvedUrl' => $url,
                ],
            ];
            return $res;
        }
        catch (\Exception $e)
        {
            print $e;
            exit;
        }
    }

    public function confirmPayment()
    {
        if (!auth()->check())
        {
            $getLoggedInCredentials = Session::get('credentials');
            if (Auth::attempt($getLoggedInCredentials))
            {
                $this->setDefaultSessionValues();
                $successPath = $this->storePaymentInformations();
                // dd($successPath);
                return redirect()->to($successPath);
            }
            else
            {
                $this->helper->one_time_message('error', __('Unable to login with provided credentials!'));
                return redirect()->back();
            }
        }
        $this->setDefaultSessionValues();
        $successPath = $this->storePaymentInformations();
        return redirect()->to($successPath);
    }

    public function cancelPayment()
    {
        $transInfo     = Session::get('transInfo');
        $trans         = AppTransactionsInfo::find($transInfo->id);
        $trans->status = 'cancel';
        $trans->save();
        Session::forget('transInfo');
        return redirect()->to($trans->cancel_url);
    }

    protected function storePaymentInformations()
    {
        $transInfo = Session::get('transInfo');
        // dd($transInfo);

        $unique_code = unique_code();
        $amount      = $transInfo->amount;
        $currency    = $transInfo->currency;
        $curr        = Currency::where('code', $currency)->first(['id']);
        if (!$curr)
        {
            $this->helper->one_time_message('error', "Currency not found!");
            return redirect()->to('payment/fail');
        }
        $senderWallet = Wallet::where(['user_id' => auth()->user()->id, 'currency_id' => $curr->id])->first();
        if ($senderWallet->balance < $amount)
        {
            $this->helper->one_time_message('error', "Doesn't have sufficient balance!");
            return redirect()->to('payment/fail');
        }
        $senderWallet->balance = $senderWallet->balance - $amount;
        $senderWallet->save();

        $p_calc = ($transInfo->app->merchant->fee * $amount) / 100;
        $p_calc = number_format((float) $p_calc, 2, '.', ''); //fix

        // Add on merchant
        $merchantPayment                    = new MerchantPayment();
        $merchantPayment->merchant_id       = $transInfo->app->merchant_id;
        $merchantPayment->user_id           = auth()->user()->id;
        $merchantPayment->gateway_reference = $unique_code;
        $merchantPayment->currency_id       = $curr->id;
        $merchantPayment->payment_method_id = 1;
        $merchantPayment->uuid              = $unique_code;
        $merchantPayment->charge_percentage = $p_calc;
        $merchantPayment->charge_fixed      = 0;
        $merchantPayment->amount            = $amount - $p_calc;
        $merchantPayment->total             = $amount;
        $merchantPayment->status            = 'Success';
        $merchantPayment->item_name         = '';
        $merchantPayment->order_no          = '';
        // dd($merchantPayment);
        $merchantPayment->save();

        $transaction_A                           = new Transaction();
        $transaction_A->user_id                  = auth()->user()->id;
        $transaction_A->end_user_id              = $transInfo->app->merchant->user_id;
        $transaction_A->currency_id              = $curr->id;
        $transaction_A->payment_method_id        = 1;
        $transaction_A->uuid                     = $unique_code;
        $transaction_A->transaction_reference_id = $merchantPayment->id;
        $transaction_A->transaction_type_id      = Payment_Sent;
        $transaction_A->subtotal                 = $amount;
        $transaction_A->percentage               = $transInfo->app->merchant->fee;
        $transaction_A->charge_percentage        = 0;
        $transaction_A->charge_fixed             = 0;
        $transaction_A->total                    = '-' . ($merchantPayment->charge_percentage + $merchantPayment->amount);
        $transaction_A->status                   = 'Success';
        $transaction_A->merchant_id              = $transInfo->app->merchant_id;
        $transaction_A->save();

        //updating merchant wallet
        $merchantWallet          = Wallet::where(['user_id' => $transInfo->app->merchant->user_id, 'currency_id' => $curr->id])->first();
        $merchantWallet->balance = ($merchantWallet->balance + $amount);
        $merchantWallet->save();

        $transaction_B                           = new Transaction();
        $transaction_B->user_id                  = $transInfo->app->merchant->user_id;
        $transaction_B->end_user_id              = auth()->user()->id;
        $transaction_B->currency_id              = $curr->id;
        $transaction_B->payment_method_id        = 1;
        $transaction_B->uuid                     = $unique_code;
        $transaction_B->transaction_reference_id = $merchantPayment->id;
        $transaction_B->transaction_type_id      = Payment_Received;
        $transaction_B->subtotal                 = $amount - ($p_calc);
        $transaction_B->percentage               = $transInfo->app->merchant->fee; //fixed
        $transaction_B->charge_percentage        = $p_calc;
        $transaction_B->charge_fixed             = 0;
        $transaction_B->total                    = $merchantPayment->charge_percentage + $merchantPayment->amount;
        $transaction_B->status                   = 'Success';
        $transaction_B->merchant_id              = $transInfo->app->merchant_id;
        $transaction_B->save();
        $transInfo->status = 'success';
        $transInfo->save();

        //pass the response to success url
        $response = [
            'status'         => 200,
            'transaction_id' => $merchantPayment->uuid,
            'merchant'       => $merchantPayment->merchant->user->first_name . ' ' . $merchantPayment->merchant->user->last_name,
            'currency'       => $merchantPayment->currency->code,
            'fee'            => $merchantPayment->charge_percentage,
            'amount'         => $merchantPayment->amount,
            'total'          => $merchantPayment->total,
        ];
        $response        = json_encode($response);
        $encodedResponse = base64_encode($response);

        // auth()->logout(); //logout of existing domain to get back to merchant domain

        $successPath = $transInfo->success_url . '?' . $encodedResponse;
        return $successPath;
    }
}
